/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xri.c 
 *       RI (Requester-Initiator) control functions
 * -----------------------------------------------------------------*/

#include <xpciapi.h>
#include <xaddrmap.h>

/* Pointer to RI database */
#define DB_RI (bx_handlearray[handle].db->Exerciser.RequesterInitiator)

/* Value of generic property */
#define GENERIC(prop) (DB_RI.GenProp[prop])

/* Pointer to one line in block-memory */
#define BLOCK(line) (DB_RI.Block.Mem[line])

/* Pointer to Block Update Info */
#define BLOCK_UPD (&DB_RI.Block.UpdateInfo)

/* Pointer to one line in behavior-memory */
#define BEHAVIOR(line) (DB_RI.Behavior.Mem[line])

/* Pointer to Behavior Update Info */
#define BEH_UPD (&DB_RI.Behavior.UpdateInfo)

/* memory maps moved to xdyndata.c */

/* the following lines leaves running code running.... */
extern bx_memmaptype Mephisto_RI_Blk2Hw [];
#define Blk2Hw Mephisto_RI_Blk2Hw

/* the following lines leaves running code running.... */
extern bx_memmaptype Mephisto_RI_Beh2Hw [];
#define Beh2Hw Mephisto_RI_Beh2Hw

/*****************************************************************************
 * Requester-Initiator Programming
 ****************************************************************************/

 /*---------------------------------------------------------------------------*
 * BestXRIProg
 *
 * Purpose: Program properties and memories from host to card
 *---------------------------------------------------------------------------*/
bx_errtype BestXRIProg (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXRIProg [riprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
  
    BX_TRY(BestXRIBlockProg(handle));
    BX_TRY(BestXRIBehProg(handle));
    BX_TRY(BestXRIGenProg(handle));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIRead
 *
 * Purpose: Read properties and memories from card- to host memory 
 *---------------------------------------------------------------------------*/
bx_errtype BestXRIRead (bx_handletype handle, bx_int32 option)
{
  BX_DECLARE_FUNCNAME("BestXRIRead [riread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY(BestXRIBlockRead(handle,option));
    BX_TRY(BestXRIBehRead(handle,option));
    BX_TRY(BestXRIGenRead(handle));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/********************************************************************
  RI generics *******************************************************
********************************************************************/

/*---------------------------------------------------------------------------*
 * BestXRIGenDefaultSet
 *
 * Purpose: Set all generic properties to default (on host only) 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRIGenDefaultSet(
  bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXRIGenDefaultSet [rigendefset]");

  BX_TRY_VARS;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_RI_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_RI_GEN,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXRIGenSet(handle,
              ParamInfo->proptyp.rigenprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIGenGet 
 *
 * Purpose: Gets a generic property from host-memory 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRIGenGet(                     
  bx_handletype        handle,
  bx_rigentype          prop,                      
  bx_int32             *val                       
)
{
  BX_DECLARE_FUNCNAME("BestXRIGenGet [rigenget]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_RI_GEN, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
     
    switch (prop)
    {
      case BX_RIGEN_NUMBEH:

        /* BX_RIGEN_NUMBEH does not exist in HW.
           If user gets this property, he
           implicitly gets it from Group 0
           (all groups usually have same value):
        */
        BX_TRY(BestXRIGenGet(handle, BX_RIGEN_NUMBEHG0, val));
        break;

      default:
        /* get property from host DB */
        *val=(bx_int32)GENERIC(prop);
        break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}
 
/*---------------------------------------------------------------------------*
 * BestXRIGenSet         
 *
 * Purpose: Sets a generic property (on host only) 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIGenSet(          
  bx_handletype        handle,
  bx_rigentype          prop,                      
  bx_int32             val                        
)
{
  BX_DECLARE_FUNCNAME("BestXRIGenSet [rigenset]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_RI_GEN, (bx_int32)prop, val));
     
    switch (prop)
    {
      case BX_RIGEN_NUMBEH:

        /* BX_RIGEN_NUMBEH does not exist in HW.
           If user sets this property, he
           implicitly sets all groups:
        */
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG0, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG1, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG2, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG3, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG4, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG5, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG6, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG7, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG8, val));
        BX_TRY(BestXRIGenSet(handle, BX_RIGEN_NUMBEHG9, val));    
        break;

      default:
        /* set property in host DB */
        GENERIC(prop)=val;
        break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIGenProg
 *
 * Purpose: Writes all generic properties to card
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRIGenProg(bx_handletype  handle)
{
  /* Updates Generics memory on card */

  BX_DECLARE_FUNCNAME("BestXRIGenProg [rigenprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 value=0, regval=0;

    #ifdef HOOK_BESTX_RIGEN_PROG_BEGIN
      HOOK_BESTX_RIGEN_PROG_BEGIN ;
    #endif

    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
        
    /* Number of block lines */
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBLK, &value));
    assert (value > 0);
    /* Value is in HW one less than number of lines */
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_CTR_LOOP_REG,sizeof(bx_int16),value-1));

    /* Repeat Block */
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_REPEATBLK, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_REP_REG, sizeof(bx_int32), value));

    /* Number of behaviors */
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG0, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR0_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG1, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR1_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG2, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR2_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG3, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR3_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG4, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR4_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG5, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR5_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG6, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR6_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG7, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR7_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG8, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR8_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_NUMBEHG9, &value));
    assert (value > 0);
    value--;
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBEHAV_CTR9_LOOP_REG, sizeof(bx_int16), value));

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_SKIPREG1, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_SKIP_ADDR_1, sizeof(bx_int16), value));
    
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_SKIPREG2, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_SKIP_ADDR_2, sizeof(bx_int16), value));
    
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_SKIPREG3, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_SKIP_ADDR_3, sizeof(bx_int16), value));
    
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_SKIPREG4, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_SKIP_ADDR_4, sizeof(bx_int16), value));
    
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_SKIPREG5, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_SKIP_ADDR_5, sizeof(bx_int16), value));
   
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_SKIPREG6, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_SKIP_ADDR_6, sizeof(bx_int16), value));
    
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_SKIPREG7, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MBLOCK_SKIP_ADDR_7, sizeof(bx_int16), value)); 

    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_TABORT, &value));
    regval = (value == BX_RIGEN_TABORT_SKIP ? 2UL : 0UL);
    BX_TRY(BestXRIGenGet (handle, BX_RIGEN_IABORT, &value));
    regval |= (value == BX_RIGEN_IABORT_SKIP ? 1UL : 0UL);
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MSM_ABORT_REG, sizeof(bx_int16), regval));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));  

    #ifdef HOOK_BESTX_RIGEN_PROG_END
      HOOK_BESTX_RIGEN_PROG_END ;
    #endif
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIGenRead
 *
 * Purpose: Reads all generic properties from card- to host-memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRIGenRead(bx_handletype  handle)
{
  /* Updates generics memory on host DB */

  BX_DECLARE_FUNCNAME("BestXRIGenRead [rigenread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 value=0;

    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
        
    /* Number of block lines */
    /* Value is in HW one less than number of lines in CAPI */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_CTR_LOOP_REG,sizeof(bx_int16),&value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBLK, value+1));

    /* Repeat Block */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_REP_REG, sizeof(bx_int32), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_REPEATBLK, value));

    /* Number of behaviors */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR0_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG0, value+1));
    
    /* NUMBEH does not exist in HW, we take group 0 (arbitrary) for it */
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEH, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR1_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG1, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR2_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG2, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR3_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG3, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR4_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG4, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR5_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG5, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR6_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG6, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR7_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG7, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR8_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG8, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR9_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_NUMBEHG9, value+1));

    /* Skip registers */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_SKIP_ADDR_1, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_SKIPREG1, value));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_SKIP_ADDR_2, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_SKIPREG2, value));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_SKIP_ADDR_3, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_SKIPREG3, value));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_SKIP_ADDR_4, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_SKIPREG4, value));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_SKIP_ADDR_5, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_SKIPREG5, value));
   
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_SKIP_ADDR_6, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_SKIPREG6, value));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_SKIP_ADDR_7, sizeof(bx_int16), &value)); 
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_SKIPREG7, value));

    /* Abort */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_MSM_ABORT_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_TABORT, (value&2 ? 1:0)));
    BX_TRY(BestXRIGenSet (handle, BX_RIGEN_IABORT, (value&1 ? 1:0)));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));  
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
 * RI Block Functions
 *****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXRIBlockMemInit
 *
 * Purpose: Set the whole block memory to default (on host only)
 * Caution: A following ExerciserProg() will re-program whole memory !!
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBlockMemInit(
    bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXRIBlockMemInit [riblockmeminit]");

  BX_TRY_VARS;
  bx_int32 offset;
  
  BX_TRY_BEGIN
  {
    for (offset=0;offset<BX_RIBLK_MEMDEPTH;offset++)
    {
      BX_TRY(BestXRIBlockDefaultSet(handle,offset));
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXRIBlockDefaultSet
 *
 * Purpose: Defaults one line in block-memory (on host only)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBlockDefaultSet(
  bx_handletype handle,
  bx_int32 offset)
{
  BX_DECLARE_FUNCNAME("BestXRIBlockDefaultSet [riblockdefset]");

  BX_TRY_VARS;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_RI_GEN, (bx_int32)BX_RIGEN_NUMBLK, offset+1));

    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_RI_BLOCK, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_RI_BLOCK,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXRIBlockSet(handle,offset,
              ParamInfo->proptyp.riblkprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBlockGet
 *
 * Purpose: Gets one block property (from host-memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBlockGet(
  bx_handletype handle,
  bx_int32 offset,
  bx_riblktype prop,
  bx_int32 * val)
{
  BX_DECLARE_FUNCNAME("BestXRIBlockGet [riblockget]");
  bx_int32 i;
  bx_int32 ival;  /* partial bits of property value */

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    #ifdef HOOK_BESTX_RIBLOCK_GET
      HOOK_BESTX_RIBLOCK_GET ;
    #endif

    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_RI_GEN, (bx_int32)BX_RIGEN_NUMBLK, offset+1));

    /* Get pointer to i-th (existing) property */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_RI_BLOCK, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    /* Walk through all entries in table Blk2Hw[].
       There is at least one entry for each property. 
    */
    
    *val=0;
    i=0;
    do   /* Table Blk2Hw[] has at least one valid entry */
    {
      if (Blk2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we get corresponding bits */       
        ival=0;
        if (!BestXMemPropGet
        (
          BLOCK(offset)+Blk2Hw[i].MemBitPos/32,       /* pointer memory location to change */
          Blk2Hw[i].MemBitPos%32,                     /* first bitpos to read */           
          Blk2Hw[i].MemBitPos%32+Blk2Hw[i].PropLen-1, /* last bitpos to read */
          &ival,                                      /* returned bits */
          Blk2Hw[i].PropName,                         /* property name for debugging */
          sizeof(bx_int32)
        ))
        {
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }

        /* merge read bits of ival into *val */
        *val = *val | ival<<Blk2Hw[i].PropBitPos; /* set bits */
      }
    } while (Blk2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }

  /* Special handling needed to hide HW details in CAPI */
  switch (prop)
  { 
    case BX_RIBLK_INTADDR:
      /* CAPI property BX_RIBLK_INTADDR is a byte address (should be qword-aligned),
         whereas the corresponding HW value in Block-memory is 
         a qword address !!
      */
      *val=*val<<3; /* convert qword- to byte address */
      break;
    default:
      break;
  }

  BX_ERRETURN(BX_E_OK);
}

/*---------------------------------------------------------------------------*
 * BestXRIBlockSet
 *
 * Purpose: Set one block property (on host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBlockSet(
  bx_handletype handle,
  bx_int32 offset,
  bx_riblktype prop,
  bx_int32 val
)
{
  BX_DECLARE_FUNCNAME("BestXRIBlockSet [riblockset]");

  bx_int32 i;

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    #ifdef HOOK_BESTX_RIBLOCK_SET
      HOOK_BESTX_RIBLOCK_SET ;
    #endif

    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_RI_GEN, (bx_int32)BX_RIGEN_NUMBLK, offset+1));

    /* dynamic capability checking concerning values of parameters */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_RI_BLOCK, (bx_int32)prop, val));

    /* Special handling needed to hide HW details in CAPI */
    switch (prop)
    { 
      case BX_RIBLK_INTADDR:
        /* CAPI property BX_RIBLK_INTADDR is a byte address (should be qword-aligned),
           whereas the corresponding HW value in Block-memory is 
           a qword address !!
        */
        val=val>>3; /* convert byte- to qword address */
        break;
      default:
        break;
    }
    
    /* Walk through all entries in table Blk2Hw[].
       There is at least one entry for each property. 
    */

    i=0;
    do   /* Table Blk2Hw[] has at least one valid entry (+ delimiter entry) */
    {
      if (Blk2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we set corresponding bits */       
   
        if (BestXMemPropSet
        (
          /* in block memory () (32 bits wide) we count Bitpositions 31,..,0 
             instead of 15,..,0,31,..,16 in behavior memory (16 bits wide).
             So we have to correct this off-by-one here in the first parameter.
           */
          BLOCK(offset)+Blk2Hw[i].MemBitPos/32, /* pointer memory location to change */
          Blk2Hw[i].MemBitPos%32,               /* first bitpos to change */           
          Blk2Hw[i].MemBitPos%32+Blk2Hw[i].PropLen-1, /* last bitpos to change */
          val>>Blk2Hw[i].PropBitPos, /* relevant bits to set start at position 0 */
          Blk2Hw[i].PropName,        /* property name for debugging */
          sizeof(bx_int32)
        ))
        {
          /* success */
          /* mark this property modified */
          SetUpdateInfo(BLOCK_UPD,1,offset,Blk2Hw[i].MemBitPos/32);  
        }
        else
        {
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }
      }
    } while (Blk2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBlockProg
 *
 * Purpose: Writes any changes made in block memory (on host) to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBlockProg(bx_handletype  handle)
{
  /* Updates block memory on card */

  BX_DECLARE_FUNCNAME("BestXRIBlockProg [riblockprog]");

  BX_TRY_VARS;

  bx_bool  Touched;  /* Wether memory needs an update at all */
  bx_int32 FirstRow,LastRow,FirstCol,LastCol; /* Range to update */

  bx_int32 row,col,i;    /* Indices */

  bx_int32ptr data=NULL;  /* Block properties to be written to card */
  bx_int32 NumDWs;   /* size of data in DWs */

  /* Retrieve info about the range to update */
  GetUpdateInfo(BLOCK_UPD,&Touched,&FirstRow,&LastRow,&FirstCol,&LastCol);

  BX_TRY_BEGIN
  {
    #ifdef HOOK_BESTX_RIBLOCK_PROG_BEGIN
      HOOK_BESTX_RIBLOCK_PROG_BEGIN ;
    #endif

    if (!Touched)
    {
      /* host and card memory are identical, no update needed */
    }
    else
    {
      /* host- and card memory differ */

      /* Additional check for INTACK command.
         Problem: If (AD&0x3)+NumBytes>4, the INTACK command is executed more than once.
         Solution: Set NumBytes fix to 1 here.
      */
      for (row=FirstRow;row<=LastRow;row++)
      {
        bx_int32 buscmd=0;

        BX_TRY(BestXRIBlockGet(handle,row,BX_RIBLK_BUSCMD,&buscmd));
        if (buscmd==BX_BUSCMD_INTACK)
        {
          BX_TRY(BestXRIBlockSet(handle,row,BX_RIBLK_NUMBYTES,1));
        }
      }
      
      /* Switch to Prog-Mode (DBI clock) */
      BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

      /* Min and Max registers determine the Column (property) -range 
         to program 
      */
      assert(FirstCol<BX_RIBLK_MEMWIDTH);
      assert(LastCol<BX_RIBLK_MEMWIDTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_MIN_REG,
                 sizeof(bx_int16),FirstCol));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_MAX_REG,
                 sizeof(bx_int16),LastCol));

      /* Set starting row (i.e. offset) from which to start programming */
      assert(FirstRow<BX_RIBLK_MEMDEPTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_CTR,
                 sizeof(bx_int16),FirstRow));

      
      /* In order to use a single call to BestXDirectRegBlockWrite() 
         instead of many calls to BestXDirectRegWrite(),
         we first have to copy the needed block properties into 
         a local data array 
      */
      
      NumDWs=(LastRow-FirstRow+1) * (LastCol-FirstCol+1);
      data=(bx_int32ptr) BestXMemCalloc(NumDWs,sizeof(bx_int32));
      if (data==NULL)
      {
        BX_ERRETURN(BX_E_HOST_MEM_FULL);
      }

      /* Fill data[] with needed properties */
      i=0;
      for (row=FirstRow;row<=LastRow;row++)
      {
        for (col=FirstCol;col<=LastCol;col++)
        {
          data[i++]=DB_RI.Block.Mem[row][col];
        }
      }

      assert(i==NumDWs);

      /* Write to master block memory */
      BX_TRY(BestXDirectRegBlockWrite(handle,
                                      BX_REG_MBLOCK_DATA_REG,
                      sizeof(bx_int32),
                                      1, /* 1=autoincrement */
                                      (bx_int8ptr) data,
                                      sizeof(bx_int32),
                                      NumDWs*sizeof(bx_int32)
                                      ));

      BestXMemFree((void**) &data);
    
      /* Mark host memory to be identical with HW */
      SetUpdateInfo(BLOCK_UPD,0,0,0);
    }

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));

  
  
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        if (data)
        {
          BestXMemFree((void**) &data);
        }
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBlockRead
 *
 * Purpose: Reads the card's block memory to host
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRIBlockRead(bx_handletype  handle,bx_int32 option)
{
  /* Updates block memory on host DB */

  BX_DECLARE_FUNCNAME("BestXRIBlockRead [riblockread]");

  BX_TRY_VARS;

  bx_int32 NumLines=0;   /* number of lines to read */

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    /* Min and Max registers determine the Column (property) -range 
       to read from
    */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_MIN_REG,
                   sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_MAX_REG,
                   sizeof(bx_int16),BX_RIBLK_MEMWIDTH-1));

    /* Set starting row (i.e. offset) from which to start reading */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBLOCK_CTR,sizeof(bx_int16),0));

    /* Determine number of lines to read */
    switch (option)
    {
      case BX_EXEREAD_CARD:
        /* 
           The number of lines to read is determined 
           by current HW values.
        */    
        BX_TRY(BestXDirectRegRead(handle, BX_REG_MBLOCK_CTR_LOOP_REG,sizeof(bx_int16),&NumLines));
        NumLines++; /* HW value is off by one */
        break;
      case BX_EXEREAD_ALL:
        /* Read complete memory */
        NumLines=BX_RIBLK_MEMDEPTH;
        break;
      case BX_EXEREAD_NONE:
        /* Read nothing */
        NumLines=0;
        break;
      case BX_EXEREAD_USER:
        /* 
           User can determine how many lines are read from card 
           by setting corresponding generic property before.
           Remark:
           During BestXExerciserRead(), the ..GenRead() 
           function must be called after this function,
           because it would overwrites the property with HW-values !!!
         */
        BX_TRY(BestXRIGenGet(handle, BX_RIGEN_NUMBLK, &NumLines));
        break;
      default:
        BX_E_ERROR_MSG_SET("BestXExerciserRead: Invalid value for parameter option");
        BX_TRY_ERROR(BX_E_ERROR);
        break;
    }


    /* Read block memory from card */
    BX_TRY(BestXDirectRegBlockRead(handle,
                                    BX_REG_MBLOCK_DATA_REG,
                    sizeof(bx_int32),
                                    1, /* 1=autoincrement */
                                    (bx_int8ptr)DB_RI.Block.Mem,
                                    sizeof(bx_int32),
                                    NumLines*BX_RIBLK_MEMWIDTH*sizeof(bx_int32)
                                    ));
    

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  /* chris: 
     Do not change the update info here, because BestXOpen()
     marks the whole memory valid (except first line).
     SetUpdateInfo(BLOCK_UPD,0,0,0);
  */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBlockCopyLine
 *
 * Purpose: Copies one block line to another.
 *          Used only by application.
 *          Do not document.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRIBlockCopyLine(
  bx_handletype  handle,
  bx_int32 srcline,
  bx_int32 destline
)
{
  BX_DECLARE_FUNCNAME("BestXRIBlockCopyLine [riblkcopyline]");
  
  bx_int32 col;
 
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    assert(srcline<BX_RIBLK_MEMDEPTH);    
    assert(destline<BX_RIBLK_MEMDEPTH);    

    /* Set the destination line to default, i.e.
       touch it, so that it will be programmed to card
       by ExerciserProg().
    */
    BX_TRY(BestXRIBlockDefaultSet(handle,destline));

    /* Copy line directly in DB */
    for (col=0;col<BX_RIBLK_MEMWIDTH;col++)
    {
      BLOCK(destline)[col]=BLOCK(srcline)[col];
    }
  }
 
  BX_ERRETURN(BX_TRY_RET);
}


/* --------------------------------------------------------------------------
 * RI Behavior Functions
 * ------------------------------------------------------------------------- */

/*---------------------------------------------------------------------------*
 * BestXRIBehProg
 *
 * Purpose: Writes changes of behavior memory from host to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBehProg(bx_handletype  handle)
{
  /* Updates master behavior on card */

  BX_DECLARE_FUNCNAME("BestXRIBehProg [ribehprog]");

  BX_TRY_VARS;

  bx_bool  Touched;  /* Wether memory needs an update at all */
  bx_int32 FirstRow,LastRow,FirstCol,LastCol; /* Range to update */

  bx_int32 row,col,i;    /* Indices */

  bx_int16ptr data=NULL;    /* part of behavior memoryto be written to card */
  bx_int32 NumWords;   /* size of data[] in words */
  
  
  /* Retrieve info about the range to update */
  GetUpdateInfo(BEH_UPD,&Touched,&FirstRow,&LastRow,&FirstCol,&LastCol);
  
  BX_TRY_BEGIN
  {
    #ifdef HOOK_BESTX_RIBEH_PROG_BEGIN
      HOOK_BESTX_RIBEH_PROG_BEGIN ;
    #endif
  
    if (!Touched)
    {
      /* host and card memory are identical, no update needed */
    }
    else
    {
      /* host- and card memory differ */

      /* Switch to Prog-Mode (DBI clock) */
      BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

      /* Min and Max registers determine the Column (property) -range 
         to program 
      */
      assert(FirstCol<BX_RIBEH_MEMWIDTH);
      assert(LastCol<BX_RIBEH_MEMWIDTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEHAV_MIN_REG,sizeof(bx_int16),FirstCol));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEHAV_MAX_REG,sizeof(bx_int16),LastCol));

      /* Set starting row (i.e. offset) from which to start programming */
      assert(FirstRow<BX_RIBEH_MEMDEPTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEHAV_CTR,sizeof(bx_int16),FirstRow));

      /* In order to use a single call to BestXDirectRegBlockWrite() 
         instead of many calls to BestXDirectRegWrite(),
         we first have to copy the needed behavior properties into 
         a local data array 
      */
      
      NumWords=(LastRow-FirstRow+1) * (LastCol-FirstCol+1);
      data=(bx_int16ptr) BestXMemCalloc(NumWords,sizeof(bx_int16));
      if (data==NULL)
      {
        BX_ERRETURN(BX_E_HOST_MEM_FULL);
      }

      /* Fill data[] with needed properties */
      i=0;
      for (row=FirstRow;row<=LastRow;row++)
      {
        for (col=FirstCol;col<=LastCol;col++)
        {
          data[i++]=DB_RI.Behavior.Mem[row][col];
        }
      }

      assert(i==NumWords);

      /* Write to master behavior memory */
      BX_TRY(BestXDirectRegBlockWrite(handle,
                                      BX_REG_MBEHAV_DATA_REG,
                      sizeof(bx_int16),
                                      1, /* 1=autoincrement */
                                      (bx_int8ptr) data,
                                      sizeof(bx_int16),
                                      NumWords*sizeof(bx_int16)
                                      ));

      BestXMemFree((void**) &data);

      /* Mark host memory to be identical with HW */
      SetUpdateInfo(BEH_UPD,0,0,0);
    }

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        if (data)
        {
          BestXMemFree((void**) &data);
        }
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBehMemInit
 *
 * Purpose: Defaults the complete behavior memory (on host only)
 * Caution: A following ExerciserProg() will re-program whole memory !!
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBehMemInit(
    bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXRIBehMemInit [ribehmeminit]");

  BX_TRY_VARS;
  bx_int32 offset;
  
  BX_TRY_BEGIN
  {
    for (offset=0;offset<BX_RIBEH_MEMDEPTH;offset++)
    {
      BX_TRY(BestXRIBehDefaultSet(handle,offset));
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBehDefaultSet
 *
 * Purpose: Defaults one behavior line (on host)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBehDefaultSet(
  bx_handletype handle,
  bx_int32 offset)
{
  BX_DECLARE_FUNCNAME("BestXRIBehDefaultSet [ribehdefset]");

  BX_TRY_VARS;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* license checking */
    BX_TRY_LICENSE(BX_CAPABILITY_CAPI); 

    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_RI_BEH, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_RI_BEH,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXRIBehSet(handle,offset,
              ParamInfo->proptyp.ribehprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBehGet
 *
 * Purpose: Gets a behavior property (from host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBehGet(
  bx_handletype handle,
  bx_int32 offset,
  bx_ribehtype prop,
  bx_int32 * val)
{
  BX_DECLARE_FUNCNAME("BestXRIBehGet [ribehget]");

  bx_int32 i;
  bx_int32 ival;  /* partial bits of property value */

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    #ifdef HOOK_BESTX_RIBEH_GET
      HOOK_BESTX_RIBEH_GET ;
    #endif

    /* license checking */
    BX_TRY_LICENSE(BX_CAPABILITY_CAPI); 

    /* Get pointer to i-th (existing) property */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_RI_BEH, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    /* Walk through all entries in table Beh2Hw[].
       There is at least one entry for each property. 
    */
    
    *val=0;
    i=0;
    do   /* Table Beh2Hw[] has at least one valid entry */
    {
      if (Beh2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we get corresponding bits */       
        ival=0;
        if (!BestXMemPropGet
        (
          (bx_int32*) (BEHAVIOR(offset)+Beh2Hw[i].MemBitPos/16),    /* pointer memory location to change */
          Beh2Hw[i].MemBitPos%16,                     /* first bitpos to read */           
          Beh2Hw[i].MemBitPos%16+Beh2Hw[i].PropLen-1, /* last bitpos to read */
          &ival,                                      /* returned bits */
          Beh2Hw[i].PropName,                         /* property name for debugging */
          sizeof(bx_int16)
        ))
        {
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }

        /* merge read bits of ival into *val */
        *val = *val | ival<<Beh2Hw[i].PropBitPos; /* set bits */
      }
    } while (Beh2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }
  
  /* We need some modifications to adapt sw to hw */
  switch (prop)    
  {
    case BX_RIBEH_REPEAT:
      if (*val==0)
      {
        /* This register is 8 bit wide in HW.
           A value of zero in HW means 256 in CAPI.
        */
        *val=0x100UL;
      }
      break;
    
    default:
      break;
  }
  
  BX_ERRETURN(BX_E_OK);
}

/*---------------------------------------------------------------------------*
 * BestXRIBehSet
 *
 * Purpose: Sets a behavior property (host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBehSet(
  bx_handletype handle,
  bx_int32 offset,
  bx_ribehtype prop,
  bx_int32 val
)

{
  BX_DECLARE_FUNCNAME("BestXRIBehSet [ribehset]");
  bx_int32 i;

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    #ifdef HOOK_BESTX_RIBEH_SET
      HOOK_BESTX_RIBEH_SET ;
    #endif

    /* dynamic capability checking concerning values of parameters */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_RI_BEH, (bx_int32)prop, val));
    
    /* We need some modifications to adapt sw to hw */
    switch (prop)        
    {
      case BX_RIBEH_REPEAT:
        if (val==0x100UL)
        {
          /* This register is 8 bit wide in HW.
             A value of zero in HW means 256 in CAPI
          */
      val=0UL;
        }
        break;
      
      default:
        break;
    }


    /* Walk through all entries in table Beh2Hw[].
       There is at least one entry for each property. 
    */

    i=0;
    do   /* Table Beh2Hw[] has at least one valid entry */
    {
      if (Beh2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we set corresponding bits */       
 
        if (BestXMemPropSet
        (
          (bx_int32*)(BEHAVIOR(offset)+Beh2Hw[i].MemBitPos/16), /* pointer memory location to change */
          Beh2Hw[i].MemBitPos%16,               /* first bitpos to change */           
          Beh2Hw[i].MemBitPos%16+Beh2Hw[i].PropLen-1, /* last bitpos to change */
          val>>Beh2Hw[i].PropBitPos, /* relevant bits to set start at position 0 */
          Beh2Hw[i].PropName,        /* property name for debugging */
          sizeof(bx_int16)
        ))
        {
          /* success */
          /* mark this property modified */
          SetUpdateInfo(BEH_UPD,1,offset,Beh2Hw[i].MemBitPos/16);  
        }
        else
        {
           
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }
      }
    } while (Beh2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBehRead
 *
 * Purpose: Reads behavior memory from card to host
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRIBehRead(bx_handletype  handle,bx_int32 option)
{
  /* Updates behavior memory on host DB */

  BX_DECLARE_FUNCNAME("BestXRIBehRead [ribehread]");

  BX_TRY_VARS;

  bx_int32 NumLines=0;   /* number of lines to read */

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    /* Min and Max registers determine the Column (property) -range 
       to read from
    */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEHAV_MIN_REG,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEHAV_MAX_REG,sizeof(bx_int16),BX_RIBEH_MEMWIDTH-1));

    /* Set starting row (i.e. offset) from which to start reading */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MBEHAV_CTR,sizeof(bx_int16),0));

    /* Determine number of lines to read */
    switch (option)
    {
      case BX_EXEREAD_CARD:
        /* 
           The number of lines to read is determined 
           by current HW values.
        */    
        BX_TRY(BestXDirectRegRead(handle, BX_REG_MBEHAV_CTR0_LOOP_REG,sizeof(bx_int16),&NumLines));
        NumLines++; /* HW value is off by one */
        break;
      case BX_EXEREAD_ALL:
        /* Read complete memory */
        NumLines=BX_RIBEH_MEMDEPTH;
        break;
      case BX_EXEREAD_NONE:
        /* Read nothing */
        NumLines=0;
        break;
      case BX_EXEREAD_USER:
        /* 
           User can determine how many lines are read from card 
           by setting corresponding generic property before.
           Remark:
           During BestXExerciserRead(), the ..GenRead() 
           function must be called after this function,
           because it would overwrites the property with HW-values !!!
         */
        BX_TRY(BestXRIGenGet(handle, BX_RIGEN_NUMBEH, &NumLines));
        break;
      default:
        BX_E_ERROR_MSG_SET("BestXExerciserRead: Invalid value for parameter option");
        BX_TRY_ERROR(BX_E_ERROR);
        break;
    }



    /* Read behavior memory from card */
    BX_TRY(BestXDirectRegBlockRead(handle,
                                    BX_REG_MBEHAV_DATA_REG,
                    sizeof(bx_int16),
                                    1, /* 1=autoincrement */
                                    (bx_int8ptr)DB_RI.Behavior.Mem,
                                    sizeof(bx_int16),
                                    NumLines*BX_RIBEH_MEMWIDTH*sizeof(bx_int16)
                                    ));


    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  /* chris: 
     Do not change the update info here, because BestXOpen()
     marks the whole memory valid (except first line).
     SetUpdateInfo(BEH_UPD,0,0,0);
  */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRIBehCopyLine
 *
 * Purpose: Copies one behavior line to another.
 *          Used only by application.
 *          Do not document.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRIBehCopyLine(
  bx_handletype  handle,
  bx_int32 srcline,
  bx_int32 destline
)
{
  BX_DECLARE_FUNCNAME("BestXRIBehCopyLine [ribehcopyline]");
  
  bx_int32 col;
 
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    assert(srcline<BX_RIBEH_MEMDEPTH);    
    assert(destline<BX_RIBEH_MEMDEPTH);    

    /* Set the destination line to default, i.e.
       touch it, so that it will be programmed to card
       by ExerciserProg().
    */
    BX_TRY(BestXRIBehDefaultSet(handle,destline));

    /* Copy line directly in DB */
    for (col=0;col<BX_RIBEH_MEMWIDTH;col++)
    {
      BEHAVIOR(destline)[col]=BEHAVIOR(srcline)[col];
    }
  }
 
  BX_ERRETURN(BX_TRY_RET);
}
